#!/bin/bash

# Determine how to run ffmpeg
# On Apple Silicon with macOS < 12, use Rosetta 2 (arch -x86_64)
FFMPEG_CMD="./ffmpeg"
if [ "$(uname -m)" = "arm64" ]; then
    MACOS_VERSION=$(sw_vers -productVersion | cut -d. -f1)
    if [ "$MACOS_VERSION" -lt 12 ]; then
        echo "Detected Apple Silicon with macOS $MACOS_VERSION, using Rosetta 2 for ffmpeg"
        FFMPEG_CMD="arch -x86_64 ./ffmpeg"
    fi
fi

# Check if arguments are provided
if [ $# -lt 2 ]; then
    echo "Usage: $0 <input_file> <output_directory>"
    exit 1
fi

INPUT_FILE="$1"
OUTPUT_DIR="$2"

# Check if input file exists
if [ ! -f "$INPUT_FILE" ]; then
    echo "Error: Input file '$INPUT_FILE' does not exist"
    exit 2
fi

# Get all information from FFmpeg in one call and store it
FFMPEG_INFO=$($FFMPEG_CMD -i "$INPUT_FILE" 2>&1)

echo "File info:"
echo "$FFMPEG_INFO"

# Get all subtitle stream lines
SUBTITLE_STREAMS=$(echo "$FFMPEG_INFO" | grep "Stream.*Subtitle")

# Count subtitles
NUM_SUBS=$(echo "$SUBTITLE_STREAMS" | wc -l)

if [ $NUM_SUBS -eq 0 ]; then
    echo "No subtitle tracks found in $INPUT_FILE"
    exit 3
fi

echo "Found $NUM_SUBS subtitle tracks in $INPUT_FILE"

# Process each subtitle stream line
echo "$SUBTITLE_STREAMS" | while IFS= read -r STREAM_LINE; do
    # Extract stream index
    STREAM_INDEX=$(echo "$STREAM_LINE" | sed -n 's/.*Stream #0:\([0-9]*\).*/\1/p')
    
    # Extract language code (in parentheses)
    if [[ "$STREAM_LINE" =~ Stream\ #0:[0-9]+\(([a-z]{3})\) ]]; then
        LANG="${BASH_REMATCH[1]}"
    else
        LANG="und"
    fi
    
    # Determine if original or not (for disposition)
    DISPOSITION="dub"
    if [[ "$STREAM_LINE" =~ \(original\) ]]; then
        DISPOSITION="original"
    fi
    
    # Extract title if it exists from metadata
    TITLE=""
    NEXT_LINE_METADATA=$(echo "$FFMPEG_INFO" | grep -A 5 "$STREAM_LINE" | grep -v "$STREAM_LINE" | grep "title")
    if [[ "$NEXT_LINE_METADATA" =~ title[[:space:]]*:[[:space:]]*([^,]*) ]]; then
        TITLE=$(echo "${BASH_REMATCH[1]}" | tr -d '"' | tr ' ' '_')
    fi
    
    # Check if hearing impaired
    if [[ "$STREAM_LINE" =~ \(hearing\ impaired\) ]]; then
        if [ -z "$TITLE" ]; then
            TITLE="SDH"
        fi
    fi
    
    # Build output filename with full path
    FILENAME="output_${STREAM_INDEX}_${LANG}_${DISPOSITION}"
    if [ -n "$TITLE" ]; then
        FILENAME="${FILENAME}_${TITLE}"
    fi
    
    OUTPUT_FILE="$OUTPUT_DIR/${FILENAME}.srt"
    
    # Extract this subtitle track using stream index
    echo "Extracting subtitle track #$STREAM_INDEX with language $LANG to $OUTPUT_FILE"
    $FFMPEG_CMD -i "$INPUT_FILE" -map 0:$STREAM_INDEX -c:s srt "$OUTPUT_FILE"
    
    echo "Extracted subtitle track $STREAM_INDEX (${LANG}${TITLE:+ - $TITLE})"
done

echo "All subtitle tracks have been extracted successfully to $OUTPUT_DIR"
exit 0
