#!/bin/bash

# Check if arguments are provided
if [ $# -lt 3 ]; then
    echo "Usage: $0 <input_video_file> <subtitles_directory> <output_video_file>"
    exit 1
fi

INPUT_VIDEO="$1"
SUBTITLES_DIR="$2"
OUTPUT_VIDEO="$3"

# Check if input file exists
if [ ! -f "$INPUT_VIDEO" ]; then
    echo "Error: Input video file '$INPUT_VIDEO' does not exist"
    exit 2
fi

# Check if subtitles directory exists
if [ ! -d "$SUBTITLES_DIR" ]; then
    echo "Error: Subtitles directory '$SUBTITLES_DIR' does not exist"
    exit 3
fi

# Check if there are subtitle files in the directory
SUBTITLE_FILES=("$SUBTITLES_DIR"/*.srt)
if [ ${#SUBTITLE_FILES[@]} -eq 0 ] || [ ! -f "${SUBTITLE_FILES[0]}" ]; then
    echo "Error: No .srt subtitle files found in '$SUBTITLES_DIR'"
    exit 4
fi

echo "Found ${#SUBTITLE_FILES[@]} subtitle files in $SUBTITLES_DIR"

# Sort subtitle files by their stream index to maintain original order
SORTED_FILES=($(for f in "${SUBTITLE_FILES[@]}"; do
    # Extract the index from filename
    FILENAME=$(basename "$f")
    if [[ $FILENAME =~ output_([0-9]+)_ ]]; then
        INDEX="${BASH_REMATCH[1]}"
        echo "$INDEX:$f"
    fi
done | sort -n | cut -d: -f2))

echo "Sorted subtitle files by their original stream index"

# Build FFmpeg command
# Start with the base input file
FFMPEG_CMD="./ffmpeg -i \"$INPUT_VIDEO\""

# Add each subtitle file as an input
for SUB_FILE in "${SORTED_FILES[@]}"; do
    FFMPEG_CMD+=" -i \"$SUB_FILE\""
done

# Map all video streams from the input
FFMPEG_CMD+=" -map 0:v"

# Map all audio streams from the input
FFMPEG_CMD+=" -map 0:a"

# Set copy for all codecs
#FFMPEG_CMD+=" -c copy -c:s srt"
FFMPEG_CMD+=" -c copy"

DEFAULT_HAS_BEEN_SET="0"

# Add mapping for each subtitle file
for i in "${!SORTED_FILES[@]}"; do
    # Get just the filename without path
    SUB_FILENAME=$(basename "${SORTED_FILES[$i]}")
    
    # Extract language code and disposition from filename
    # Pattern: output_INDEX_LANG_DISPOSITION.srt or output_INDEX_LANG_DISPOSITION_TITLE.srt
    if [[ $SUB_FILENAME =~ output_[0-9]+_([a-z]{3})_(original|dub)(.*)\.srt ]]; then
        LANG="${BASH_REMATCH[1]}"
        DISPOSITION="${BASH_REMATCH[2]}"
        TITLE_PART="${BASH_REMATCH[3]}"
        
        # If title part exists, extract title (remove leading underscore)
        TITLE=""
        if [[ -n "$TITLE_PART" && "$TITLE_PART" =~ ^_(.+)$ ]]; then
            TITLE="${BASH_REMATCH[1]}"
        fi
    else
        echo "Warning: Could not parse filename format for $SUB_FILENAME, using defaults"
        LANG="und"
        DISPOSITION=""
        TITLE=""
    fi
    
    # Add subtitle mapping with language metadata
    FFMPEG_CMD+=" -map $((i+1)):0 -metadata:s:s:$i language=$LANG"
    
    # Add disposition metadata
    if [ "$DISPOSITION" == "original" ] && [ "$DEFAULT_HAS_BEEN_SET" == "0" ]; then
        FFMPEG_CMD+=" -disposition:s:$i default"
        DEFAULT_HAS_BEEN_SET="1"
    else
        FFMPEG_CMD+=" -disposition:s:$i 0"
    fi
    
    # Add title metadata if available
    if [ -n "$TITLE" ]; then
        # Replace underscores with spaces for the title
        DISPLAY_TITLE="${TITLE//_/ }"
        FFMPEG_CMD+=" -metadata:s:s:$i title=\"$DISPLAY_TITLE\""
    fi
    
    echo "Processing subtitle: Lang=$LANG, Disposition=$DISPOSITION, Title=$TITLE"
done

# Set the output file
FFMPEG_CMD+=" \"$OUTPUT_VIDEO\""

echo "Executing FFmpeg command:"
echo "$FFMPEG_CMD"

# Execute the command
eval "$FFMPEG_CMD"

if [ $? -eq 0 ]; then
    echo "Successfully merged video with subtitles to $OUTPUT_VIDEO"
    exit 0
else
    echo "Error occurred during merging"
    exit 5
fi
